# -*- coding: utf-8 -*-
import re
import unicodedata
from pathlib import Path

import pdfplumber
import pandas as pd
import logging
from datetime import datetime

# Configuración del logging
logging.raiseExceptions = False
logging.basicConfig(
    level=logging.INFO,
    format="%(asctime)s - %(levelname)s - %(message)s",
    handlers=[
        logging.FileHandler("extraccion_medidas_control.log", mode="a", encoding="utf-8"),
        logging.StreamHandler(),
    ],
)

def normalizar_texto(s: str) -> str:
    """
    Normaliza texto removiendo acentos, convirtiendo a minúsculas y eliminando puntuación.
    """
    s = unicodedata.normalize("NFD", s)
    s = "".join(ch for ch in s if unicodedata.category(ch) != "Mn")
    s = s.lower()
    s = re.sub(r"[^\w\s]", " ", s)
    return re.sub(r"\s+", " ", s).strip()

def normalizar_texto_preservar_dos_puntos(s: str) -> str:
    """
    Normalización que preserva los dos puntos para casos especiales.
    Útil para detectar frases como "medidas de control:"
    """
    s = unicodedata.normalize("NFD", s)
    s = "".join(ch for ch in s if unicodedata.category(ch) != "Mn")
    s = s.lower()
    s = re.sub(r"[^\w\s:]", " ", s)
    return re.sub(r"\s+", " ", s).strip()

# ═══════════════════════════════════════════════════════════════
# DEFINICIÓN DE PATRONES DE BÚSQUEDA (TRIGGERS)
# ═══════════════════════════════════════════════════════════════

# Verbos base que indican implementación de medidas
TRIGGER_BASE = [
    "implementara",
    "implementaran",
    "implementacion",
    "implementa",
    "aplicara",
    "aplicaran",
    "aplicacion",
    "aplica",
    "ha incorporado en su diseño una serie",
    "incorporacion en el diseño",
    "incorporacion",
    "incorpora",
    "utilizacion",
    "utilizara",
    "utiliza",
    "ejecutaran",
    "ejecucion",
    "estableceran",
    "establecen",
    "establece",
    "establecimiento",
    "contemplan",
    "contempla",
    "contemplara",
    "adoptar",
    "adopcion",
    "adoptara",
    "adoptaran",
    "consideraran",
    "considera",
    "consideran"
]

# Complementos que se refieren a medidas de control
TRIGGER_SUFFIXES = [
    "la medida de control",
    "las medidas de control",
    "medida de control",
    "medidas de control",
    "de medidas de control",
    "de las medidas de control",
    "de la medida de control",
    "medidas para el control",
    "medida para el control",
    "de las siguientes medidas de control",
    "las siguientes medidas de control",
    "las siguientes medidas para el control",
    "la siguiente medida de control",
    "la siguiente medida para el control",
]

# Construcción de TRIGGERS - todas las combinaciones posibles
TRIGGERS_1 = [f"{base} {suffix}" for base in TRIGGER_BASE for suffix in TRIGGER_SUFFIXES]

# Segundo conjunto de patrones (estructura diferente)
TRIGGER_BASE2 = [
    "medidas de control"
]
TRIGGER_SUFFIXES2 = [
    "seran las siguientes",
    "de emisiones seran las siguientes",
    "que se implementaran",
    "a implementar",
    "tales como"
]

TRIGGERS_2 = [f"{base} {suffix}" for base in TRIGGER_BASE2 for suffix in TRIGGER_SUFFIXES2]

# Trigger especial con dos puntos (requiere tratamiento diferente)
TRIGGERS_ESPECIALES = ["medidas de control:"]

# Combinar todos los triggers
TRIGGERS = list(set(TRIGGERS_1 + TRIGGERS_2))
# Normalizar triggers para búsqueda (sin espacios)
TRIGGER_PATTERNS = [normalizar_texto(t).replace(" ", "") for t in TRIGGERS]

# Crear patrones especiales para frases con dos puntos
TRIGGER_PATTERNS_ESPECIALES = []
for trigger in TRIGGERS_ESPECIALES:
    pattern_especial = normalizar_texto_preservar_dos_puntos(trigger).replace(" ", "")
    TRIGGER_PATTERNS_ESPECIALES.append((pattern_especial, trigger))

# ═══════════════════════════════════════════════════════════════
# FUNCIONES DE BÚSQUEDA Y EXTRACCIÓN
# ═══════════════════════════════════════════════════════════════

def buscar_frase_en_lineas(lineas: list, frase_normalizada: str) -> list:
    """
    Devuelve las líneas iniciales donde aparece la frase:
    - en una sola línea, o
    - cortada entre dos líneas contiguas (ej: "medidas de" en línea i y "control" en línea i+1)
    """
    # Normalizar todas las líneas (sin espacios)
    ln = [normalizar_texto(l).replace(" ", "") for l in lineas]
    hits = set()

    for i, l in enumerate(ln):
        # Caso 1: la frase está completa en una sola línea
        if frase_normalizada in l:
            hits.add(i)
            continue

        # Caso 2: la frase está cortada entre esta línea y la siguiente
        if i + 1 < len(ln):
            comb = l + ln[i + 1]
            if (frase_normalizada not in ln[i] and
                frase_normalizada not in ln[i + 1] and
                frase_normalizada in comb):
                hits.add(i)

    return sorted(hits)

def buscar_frase_especial_en_lineas(lineas: list, frase_normalizada_especial: str) -> list:
    """
    Busca "medidas de control:" solo cuando cumple condiciones específicas:
    1. Esté al inicio de la línea
    2. No haya triggers base en la línea anterior
    3. No haya más texto después de los dos puntos
    """
    ln = [normalizar_texto_preservar_dos_puntos(l).replace(" ", "") for l in lineas]
    hits = set()

    trigger_base_patterns = [normalizar_texto(trigger).replace(" ", "") for trigger in TRIGGERS]

    for i, l in enumerate(ln):
        # Verificar si la línea actual empieza con "medidasdecontrol:" y termina ahí
        if l.startswith(frase_normalizada_especial) and l == frase_normalizada_especial:
            linea_anterior_limpia = True
            if i > 0:
                linea_anterior = ln[i - 1]
                # Revisar si algún trigger base aparece en la línea anterior para evitar duplicados
                for trigger_pattern in trigger_base_patterns:
                    if trigger_pattern in linea_anterior:
                        linea_anterior_limpia = False
                        break

            if linea_anterior_limpia:
                hits.add(i)

    return sorted(hits)

def extraer_contexto(lineas: list, linea_objetivo: int, frase_encontrada: str, lineas_despues: int = 5) -> str:
    """
    Extrae todo el texto desde la frase encontrada hasta las siguientes 5 líneas.
    """
    # Obtener la línea donde se encontró la frase
    linea_actual = lineas[linea_objetivo]

    # Dividir en palabras tanto la línea como la frase
    palabras_linea = linea_actual.split()
    palabras_frase = frase_encontrada.split()

    inicio_desde = 0

    # Buscar dónde comienza la frase en la línea
    for i in range(len(palabras_linea) - len(palabras_frase) + 1):
        coincide = True
        for j in range(len(palabras_frase)):
            if i + j >= len(palabras_linea):
                coincide = False
                break
            if normalizar_texto(palabras_linea[i + j]) != normalizar_texto(palabras_frase[j]):
                coincide = False
                break

        if coincide:
            inicio_desde = i
            break

    # Extraer desde la posición encontrada hasta el final de la línea
    texto_desde_frase = ' '.join(palabras_linea[inicio_desde:])

    contexto_completo = [texto_desde_frase]

    # Determinar hasta qué línea extraer (sin exceder el total de líneas)
    fin = min(len(lineas), linea_objetivo + lineas_despues + 1)
    for i in range(linea_objetivo + 1, fin):
        contexto_completo.append(lineas[i])

    return ' '.join(contexto_completo).strip()

def procesar_pdf(ruta: Path):
    """
    Procesa un PDF buscando las frases trigger y extrae el contexto.
    """
    doc_id = ruta.stem
    registros = []

    with pdfplumber.open(ruta) as pdf:
        for num_pagina, page in enumerate(pdf.pages, 1):
            # Extraer texto de la página
            texto_pagina = page.extract_text()
            if not texto_pagina:
                continue

            # Dividir en líneas una sola vez
            lineas = texto_pagina.split('\n')

            # ──────────────────────────────────────────────────────
            # Buscar triggers normales (sin dos puntos)
            # ──────────────────────────────────────────────────────
            for i, trigger_pattern in enumerate(TRIGGER_PATTERNS):
                lineas_con_frase = buscar_frase_en_lineas(lineas, trigger_pattern)

                for linea_num in lineas_con_frase:
                    contexto = extraer_contexto(lineas, linea_num, TRIGGERS[i])

                    # Crear registro con la información
                    registro = {
                        'documento': doc_id,
                        'pagina': num_pagina,
                        'frase_encontrada': TRIGGERS[i],
                        'contexto': contexto
                    }

                    registros.append(registro)
                    logging.info(f"  Encontrado en página {num_pagina}: '{TRIGGERS[i]}'")

            # ──────────────────────────────────────────────────────
            # Buscar triggers especiales (con dos puntos)
            # ──────────────────────────────────────────────────────
            for pattern_especial, trigger_original in TRIGGER_PATTERNS_ESPECIALES:
                lineas_con_frase = buscar_frase_especial_en_lineas(lineas, pattern_especial)

                for linea_num in lineas_con_frase:
                    contexto = extraer_contexto(lineas, linea_num, trigger_original)

                    # Crear registro
                    registro = {
                        'documento': doc_id,
                        'pagina': num_pagina,
                        'frase_encontrada': trigger_original,
                        'contexto': contexto
                    }

                    registros.append(registro)
                    logging.info(f"  Encontrado en página {num_pagina}: '{trigger_original}' (especial)")

    return registros

# ═══════════════════════════════════════════════════════════════
# EJECUCIÓN PRINCIPAL
# ═══════════════════════════════════════════════════════════════

if __name__ == "__main__":
    inicio = datetime.now()
    logging.info("Proceso iniciado: %s", inicio.strftime("%Y-%m-%d %H:%M:%S"))

    # Definir carpeta donde están los PDFs (ejemplo para documentos RCA DIA)
    carpeta = Path("documentos_RCA/DIA")

    # Cargar lista de documentos a procesar desde Excel
    id_df = pd.read_excel('listado_documentos.xlsx', sheet_name='DIA', usecols=['id'])
    id_list = set(id_df['id'].astype(str).tolist())

    # Filtrar solo los PDFs que están en la lista
    pdfs = [f for f in sorted(carpeta.glob("*.pdf")) if f.stem in id_list]

    # Lista para almacenar todos los registros encontrados
    todos_los_registros = []

    # Procesar cada PDF
    for pdf_path in pdfs:
        logging.info(f"Procesando archivo número {pdfs.index(pdf_path)+1} de {len(pdfs)}: {pdf_path.name}")
        registros = procesar_pdf(pdf_path)
        todos_los_registros.extend(registros)

    df = pd.DataFrame(todos_los_registros)

    # Limpiar saltos de línea en el campo contexto
    df['contexto'] = df['contexto'].str.replace('\n', ' ', regex=False)

    # Crear resumen por documento
    resumen = df.groupby('documento').agg({
        'pagina': lambda x: ', '.join(map(str, sorted(set(x)))),
        'frase_encontrada': 'count'
    }).rename(columns={
        'pagina': 'paginas_con_matches',
        'frase_encontrada': 'total_matches'
    }).reset_index()

    # Guardar resultados en Excel
    with pd.ExcelWriter("medidas_control.xlsx", engine='openpyxl') as writer:
        df.to_excel(writer, sheet_name='Detalles', index=False)
        resumen.to_excel(writer, sheet_name='Resumen', index=False)

    logging.info("Proceso completado. Resultados guardados en 'medidas_control.xlsx'.")

    fin = datetime.now()
    logging.info("Proceso finalizado: %s", fin.strftime("%Y-%m-%d %H:%M:%S"))